﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using Pc.Policy.Smartcard.Data;
using Pc.Policy.Smartcard.Pinpad;
using Pc.Policy.Smartcard.Shared;
using Pc.Shared.BER;
using Pc.Shared.BER.Converter;
using Pc.Shared.Security.Crypto;
using Pc.Shared.Utils.Collections;
using Pc.Shared.Utils.Extensions;
using SmartCard.PCSC;
using SmartCard.PCSC.Native;

namespace Pc.Policy.Smartcard.ChangeSMO
{
    class Program
    {
        /// <summary>Точка входа в приложение</summary>
        /// <param name="args">Параметры запуска</param>
        static void Main(string[] arguments)
        {
            ArgumentCollection args = ArgumentCollection.GetArgumets(arguments);
            if (args["help"] != null || args["h"] != null)
            {
                PrintUsage();
                return;
            }
            if (args.ContainsKey("rdl"))
            {
                //Вывод списка ридеров установленных в системе
                PCSCReadersManager manager = new PCSCReadersManager();
                try
                {
                    printf("Список ридеров:");
                    //Установление соединения с смарт-карт API
                    manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                    //Вывод списка ридеров в консоль
                    manager.OfType<ISCard>().ToList().ForEach(reader =>
                    {
                        printf(reader.ReaderName);
                    });
                    manager.ReleaseContext();
                }
                catch (Exception exception)
                {
                    printf("Ошибка получения списка ридеров:\r\n", exception.Message);
                    return;
                }
            }
            if (args["csmo"] != null)
            {
                //Смена СМО
                ChangeSMO(args);
                return;
            }
            PrintUsage();
        }
        /// <summary>
        /// Данный метод выполняет смену СМО карты полиса ОМС.
        /// Для успешного выполнения данного метода требуется модуль безопасности 
        /// с соответствующими ключами шифрования разрешающими выполнение данной операции
        /// </summary>
        /// <param name="args">Параметры запуска приложения</param>
        static void ChangeSMO(ArgumentCollection args)
        {
            printf();
            printf("Смена СМО...");
            if (string.IsNullOrEmpty(args["csmo"]))
            {
                printf("Не задано имя устройства чтения смарт-карт для работы с полисом ОМС");
                return;
            }
            if (args.ContainsKey("token") == false || string.IsNullOrEmpty(args["token"]))
            {
                printf("Не задано имя устройства чтения смарт-карт для работы с модулем безопасности");
                return;
            }
            //Чтение информации о владельце карты полиса ОМС
            PCSCReadersManager manager = new PCSCReadersManager();
            try
            {
                //Установление соединения с смарт-карт API
                manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                //Вывод списка ридеров в консоль
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["csmo"]) == false)
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                    return;
                }
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["token"]) == false)
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                    return;
                }
                //Получение объекта ридера
                ISCard policy_card = manager[args["csmo"]];
                ISCard token_card = manager[args["token"]];
                //Создание объекта для работы с картой полиса ОМС
                PolicySmartcard policy = new PolicySmartcard(policy_card, token_card);
                //Подключение к карте полиса ОМС
                policy.Connect();
                //Аутентификация с картой модуля безопасности
                policy.Authentificate_Mutual(/*Пин-код модуля безопасности*/"11111111", true);
                VerifyPINArguments verify_args = new VerifyPINArguments()
                {
                    //Данный параметр устанавливает признак использования пин-пада для 
                    //предъявления пин-кода с помощью выносной клавиатуры. В случае установки данного параметра в значение false
                    //требуется заполнение параметра 'PinValue'
                    //Использование пин-пада возможно только в случае, если карта полиса ОМС установлена
                    //в устройство с поддержкой функции выносной клавиатуры.
                    //Проверить наличие функций выносной клавиатуры у устройства можно с помощью метода 
                    //'PolicyPinpad.IsDeviceSupported(policy_card, false
                    /*Не выполнять подключение к карте если подключение было выполнено внешними средствами*/
                    //)'
                    UsePinpad = PolicyPinpad.IsDeviceSupported(policy_card, false),
                    //Значение пин-кода. Данный параметр используется только если UsePinpad = false
                    PinValue = "1234",
                    //Ссылка на метод для отмены ввода пин-кода программным способом в случае использовании выносной клавиатуры
                    //Инициализация происходит на стороне метода VerifyPIN
                    CancelHandler = null,
                    //Ссылка на метод для обработки события нажатия кнопок на выносной клавиатуре
                    //Можно не инициализировать - используется только в случае использовании выносной клавиатуры
                    EnteredDigitHandler = (PinpadKey key) =>
                    {
                        //Тело метода, обработчика
                    }
                };
                policy.VerifyPIN(ref verify_args);
                //Создание объекта нового СМО
                SMOInformation smo = new SMOInformation()
                {
                    OGRN = "09876",
                    OKATO = "98765",
                    InsuranceStartDate = DateTime.Now,
                    InsuranceExpireDate = DateTime.Now.AddYears(5),
                    EDS = null /*ЭЦП будет инициализирована позже после ее создания*/
                };
                //[Формирование ЭЦП нового СМО]
                //Открытие хранилища сертификатов текущего пользователя
                X509Store store = new X509Store(StoreName.My, StoreLocation.CurrentUser);
                store.Open(OpenFlags.ReadOnly);
                //Выбор сертификата с помощью строенного диалога
                X509Certificate2Collection selected =
                  X509Certificate2UI.
                  SelectFromCollection(store.Certificates,
                  "Сертификата",
                  "Выберите сертификат для формирования ЭЦП СМО", X509SelectionFlag.SingleSelection);
                if (selected.Count > 0)
                {
                    //Создание объекта формирования подписи
                    SignatureEncoder signature_encoder = new SignatureEncoder(selected[0]);
                    //Формирование ЭЦП
                    smo.EDS = new SMOInformationEDS()
                    {
                        Signature = signature_encoder.CreateSignHash(smo.EncodeBER()/*Преобразование объекта к формату BER-TLV*/),
                        Certificate = selected[0].Export(X509ContentType.Cert)
                    };
                }
                //Запись информации о новом СМО на карту полиса ОМС
                policy.CreateNewSMO(smo);
                //Отключение от карты полиса ОМС
                policy.Disconnect();
                //Отключение от смарт-карт API
                manager.ReleaseContext();
                return;
            }
            catch (Exception exception)
            {
                printf("Ошибка чтения информации с карты:\r\n", exception.Message);
            }
        }
        /// <summary>
        /// Вывод информации о параметрах запуска приложения
        /// </summary>
        static void PrintUsage()
        {
            Console.Clear();
            printf("ЗАО Атлас-Карт © 2011. Пример работы с картой полиса ОМС.");
            printf("-rdl | Получение списка ридеров в системе");
            printf("-csmo=\"...\" -token=\"...\" | Получение списка истории СМО. Пример: -csmo=\"ACS ACR38U 0\" -token=\"\" ");
        }
        /// <summary>
        /// Вывод информационной строки в консоль
        /// </summary>
        /// <param name="format">Форматирующая строка</param>
        /// <param name="parameters">Параметры</param>
        static void printf(string format = "", params object[] parameters)
        {
            Console.WriteLine(format, parameters);
        }
    }
}
