﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using Pc.Policy.Smartcard.Crypto;
using Pc.Policy.Smartcard.Data;
using Pc.Policy.Smartcard.Shared;
using Pc.Shared.Utils.Collections;
using Pc.Shared.Utils.Extensions;
using SmartCard.PCSC;
using SmartCard.PCSC.Native;

namespace Pc.Policy.Smartcard.Read
{
    class Program
    {
        /// <summary>Точка входа в приложение</summary>
        /// <param name="args">Параметры запуска</param>
        static void Main(string[] arguments)
        {
            //Получение параметров запуска
            ArgumentCollection args = ArgumentCollection.GetArgumets(arguments);
            if (args["help"] != null || args["h"] != null)
            {
                PrintUsage();
                return;
            }
            if (args.ContainsKey("rdl"))
            {
                //Вывод списка ридеров установленных в системе
                PCSCReadersManager manager = new PCSCReadersManager();
                try
                {
                    printf("Список ридеров:");
                    //Установление соединения с смарт-карт API
                    manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                    //Вывод списка ридеров в консоль
                    manager.OfType<ISCard>().ToList().ForEach(reader =>
                    {
                        printf(reader.ReaderName);
                    });
                    manager.ReleaseContext();
                }
                catch (Exception exception)
                {
                    printf("Ошибка получения списка ридеров:\r\n", exception.Message);
                    return;
                }
            }
            if (args["rci"] != null)
            {
                //Чтение информации о карте полиса ОМС
                ReadCardInformation(args);
                return;
            }
            if (args["roi"] != null)
            {
                //Чтение информации о владельце карты полиса ОМС
                ReadOwnerInformation(args);
                return;
            }
            if (args["rcs"] != null)
            {
                //Чтение информации о текущем СМО карты полиса ОМС
                ReadCurrentSMO(args);
                return;
            }
            PrintUsage();
        }

        static void ReadCardInformation(ArgumentCollection args)
        {
            printf();
            if (string.IsNullOrEmpty(args["rci"]))
            {
                printf("Не задано имя устройства чтения смарт-карт");
                return;
            }
            //Чтение информации о владельце карты полиса ОМС
            PCSCReadersManager manager = new PCSCReadersManager();
            try
            {
                //Установление соединения с смарт-карт API
                manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                //Вывод списка ридеров в консоль
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["rci"]))
                {
                    //Получение объекта ридера
                    ISCard card = manager[args["rci"]];
                    //Создание объекта для работы с картой полиса ОМС
                    PolicySmartcardBase policy = new PolicySmartcardBase(card);
                    //Подключение к карте полиса ОМС
                    policy.Connect();
                    //Чтение информации о карте полиса ОМС ( Идентификационные данные )
                    CardID cid = policy.GetCardID();
                    //Чтение информации о карте полиса ОМС ( Информация о микросхеме от производителя )
                    ICCID ccd = policy.GetICCD();

                    if (cid != null)
                    {
                        printf("Идентификационные данные карты:");
                        printf("{0} = {1}", "Серийный номер карты".PadRight(35, ' '), cid.SerialNumber.ToHexString());
                        printf("{0} = {1}", "Тип карты".PadRight(35, ' '), cid.CardType);
                        printf("{0} = {1}", "Версия".PadRight(35, ' '), cid.CardVersion);
                        printf("{0} = {1}", "Идентификатор учреждения".PadRight(35, ' '), cid.InstitutionID.ToHexString());
                        printf("{0} = {1}", "Дополнительные сведения о карте".PadRight(35, ' '), cid.AdditionalInfo.ToHexString());
                    }
                    else
                    {
                        printf("Идентификационные данные отсутствуют на карте!");
                    }

                    if (ccd != null)
                    {
                        printf("Информация о микросхеме от производителя:");
                        printf("{0} = {1}", "Код производителя карты".PadRight(35, ' '), ccd.IssuerCode);
                        printf("{0} = {1}", "Данные производителя карты".PadRight(35, ' '), ccd.IssuerData.ToHexString());
                    }
                    else
                    {
                        printf("Информация о микросхеме от производителя отсутствует на карте!");
                    }
                    //Отключение от карты полиса ОМС
                    policy.Disconnect();
                    //Отключение от смарт-карт API
                    manager.ReleaseContext();
                    return;
                }
                else
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                }
            }
            catch (Exception exception)
            {
                printf("Ошибка чтения информации с карты:\r\n", exception.Message);
                return;
            }
        }

        //Чтение информации о владельце карты полиса ОМС
        static void ReadCurrentSMO(ArgumentCollection args)
        {
            printf();
            if (string.IsNullOrEmpty(args["rcs"]))
            {
                printf("Не задано имя устройства чтения смарт-карт");
                return;
            }
            //Чтение информации о владельце карты полиса ОМС
            PCSCReadersManager manager = new PCSCReadersManager();
            try
            {
                //Установление соединения с смарт-карт API
                manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                //Вывод списка ридеров в консоль
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["rcs"]))
                {
                    //Получение объекта ридера
                    ISCard card = manager[args["rcs"]];
                    //Создание объекта для работы с картой полиса ОМС
                    PolicySmartcardBase policy = new PolicySmartcardBase(card);
                    //Подключение к карте полиса ОМС
                    policy.Connect();
                    //Чтение информации о владельце полиса ОМС
                    SMOInformation smo = policy.GetCurrentSMOInformation();
                    if (smo != null)
                    {
                        printf("Данные страхового приложения:");
                        printf("{0} = {1}", "ОГРН".PadRight(35, ' '), smo.OGRN);
                        printf("{0} = {1}", "ОКАТО".PadRight(35, ' '), smo.OKATO);
                        printf("{0} = {1}", "Дата начала страхования".PadRight(35, ' '), FormatPolicyDate(smo.InsuranceStartDate, "Отсутствует"));
                        printf("{0} = {1}", "Окончание срока действия".PadRight(35, ' '), FormatPolicyDate(smo.InsuranceExpireDate, "Не ограничено"));
                        if (smo.EDS != null)
                        {
                            printf("Данные ЭЦП");
                            printf("{0} = {1}", "Сертификат".PadRight(35, ' '), new X509Certificate2(smo.EDS.Certificate).Subject);
                            printf("{0} = {1}", "ЭЦП".PadRight(35, ' '), smo.EDS.Signature.ToHexString());
                            printf("{0} = {1}", "Статус ЭЦП".PadRight(35, ' '), smo.VerifyEDS() == true ? "Верна" : "Неверна");
                        }
                        else
                        {
                            printf("{0} = {1}", "Данные ЭЦП".PadRight(35, ' '), "Отсутствуют");
                        }
                    }
                    else
                    {
                        printf("Данные отсутствуют на карте!");
                    }
                    //Отключение от карты полиса ОМС
                    policy.Disconnect();
                    //Отключение от смарт-карт API
                    manager.ReleaseContext();
                    return;
                }
                else
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                }
            }
            catch (Exception exception)
            {
                printf("Ошибка чтения информации с карты:\r\n", exception.Message);
                return;
            }
        }

        //Чтение информации о владельце карты полиса ОМС
        static void ReadOwnerInformation(ArgumentCollection args)
        {
            printf();
            if (string.IsNullOrEmpty(args["roi"]))
            {
                printf("Не задано имя устройства чтения смарт-карт");
                return;
            }
            //Чтение информации о владельце карты полиса ОМС
            PCSCReadersManager manager = new PCSCReadersManager();
            try
            {
                //Установление соединения с смарт-карт API
                manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                //Вывод списка ридеров в консоль
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["roi"]))
                {
                    //Получение объекта ридера
                    ISCard card = manager[args["roi"]];
                    //Создание объекта для работы с картой полиса ОМС
                    PolicySmartcardBase policy = new PolicySmartcardBase(card);
                    //Подключение к карте полиса ОМС
                    policy.Connect();
                    //Чтение информации о владельце полиса ОМС
                    OwnerInformation owner_info = policy.GetOwnerInformation();
                    if (owner_info != null)
                    {
                        printf("Информация о владельце:");
                        printf("{0} = {1}", "Фамилия".PadRight(35, ' '), FormatPolicyText(owner_info.Identity_1, "Отсутствует"));
                        printf("{0} = {1}", "Имя".PadRight(35, ' '), FormatPolicyText(owner_info.Identity_2, "Отсутствует"));
                        printf("{0} = {1}", "Отчество".PadRight(35, ' '), FormatPolicyText(owner_info.Identity_3, "Отсутствует"));
                        printf("{0} = {1}", "Пол".PadRight(35, ' '), owner_info.Sex == 1 ? "Мужской" : owner_info.Sex == 2 ? "Женский" : "Неизвестно");
                        printf("{0} = {1}", "Дата рождения".PadRight(35, ' '), FormatPolicyDate(owner_info.BirthDate, "Отсутствует"));
                        printf("{0} = {1}", "Место рождения".PadRight(35, ' '), FormatPolicyText(owner_info.BirthPlace, "Отсутствует"));
                        if (owner_info.Citizenship != null)
                        {
                            printf("Гражданство:");
                            printf("{0} = {1}", "Код страны".PadRight(35, ' '), FormatPolicyText(owner_info.Citizenship.CoutryCode, "Отсутствует"));
                            printf("{0} = {1}", "Кириллическое название".PadRight(35, ' '), FormatPolicyText(owner_info.Citizenship.CoutryCyrillicName, "Отсутствует"));
                        }
                        else
                        {
                            printf("{0} = {1}", "Гражданство".PadRight(35, ' '), "Отсутствует");
                        }
                        printf("Информация о полисе:");
                        printf("{0} = {1:D16}", "Номер полиса".PadRight(35, ' '), owner_info.PolicyNumber);
                        printf("{0} = {1}", "Дата выпуска".PadRight(35, ' '), FormatPolicyDate(owner_info.ExpireDate, "Отсутствует"));
                        printf("{0} = {1}", "Срок действия".PadRight(35, ' '), FormatPolicyDate(owner_info.ExpireDate, "Не ограничено"));
                        printf("{0} = {1}", "СНИЛС".PadRight(35, ' '), FormatPolicyText(owner_info.SNILS, "Отсутствует"));
                        printf("Чтение атрибутов безопасности карты полиса ОМС");
                        SecurityInformation SOD = policy.GetSecurityInformation();
                        printf("{0} = {1}", "Статус ЭЦП".PadRight(35, ' '), owner_info.VerifyEDS(SOD) ? "Верна" : "Неверна");
                    }
                    else
                    {
                        printf("Данные отсутствуют на карте!");
                    }
                    //Отключение от карты полиса ОМС
                    policy.Disconnect();
                    //Отключение от смарт-карт API
                    manager.ReleaseContext();
                    return;
                }
                else
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                }
            }
            catch (Exception exception)
            {
                printf("Ошибка чтения информации с карты:\r\n", exception.Message);
                return;
            }
        }

        static void PrintUsage()
        {
            Console.Clear();
            printf("ЗАО Атлас-Карт © 2011. Пример работы с картой полиса ОМС.");
            printf("-rdl = Получение списка ридеров в системе");
            printf("-roi = Прочитать информацию о владельце карты полиса ОМС.\r\n\tПример: -roi=\"ACS ACR38U 0\"");
            printf("-rсs = Прочитать информацию о текущем СМО.\r\n\tПример: -rcs=\"ACS ACR38U 0\"");
        }

        static void printf(string format = "", params object[] parameters)
        {
            Console.WriteLine(format, parameters);
        }

        static string FormatPolicyText(string value, string null_value)
        {
            if (string.IsNullOrEmpty(value))
                return null_value;
            return value;
        }

        static string FormatPolicyDate(DateTime? date, string null_value)
        {
            if (date.HasValue == false)
                return null_value;
            if (date.HasValue)
            {
                string sdate = date.Value.ToString("ddMMyyyy");
                if (sdate == "01011900")
                    return null_value;
                else
                    return date.Value.ToString("dd.MM.yyyy");
            }
            return null_value;
        }
    }
}
