﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using Pc.Policy.Smartcard.Data;
using Pc.Policy.Smartcard.Shared;
using Pc.Shared.BER;
using Pc.Shared.BER.Converter;
using Pc.Shared.Utils.Collections;
using Pc.Shared.Utils.Extensions;
using SmartCard.PCSC;
using SmartCard.PCSC.Native;

namespace Pc.Policy.Smartcard.ReadSMOHistory
{
    class Program
    {
        /// <summary>Точка входа в приложение</summary>
        /// <param name="args">Параметры запуска</param>
        static void Main(string[] arguments)
        {
            ArgumentCollection args = ArgumentCollection.GetArgumets(arguments);
            if (args["help"] != null || args["h"] != null)
            {
                PrintUsage();
                return;
            }
            if (args.ContainsKey("rdl"))
            {
                //Вывод списка ридеров установленных в системе
                PCSCReadersManager manager = new PCSCReadersManager();
                try
                {
                    printf("Список ридеров:");
                    //Установление соединения с смарт-карт API
                    manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                    //Вывод списка ридеров в консоль
                    manager.OfType<ISCard>().ToList().ForEach(reader =>
                    {
                        printf(reader.ReaderName);
                    });
                    manager.ReleaseContext();
                }
                catch (Exception exception)
                {
                    printf("Ошибка получения списка ридеров:\r\n", exception.Message);
                    return;
                }
            }
            if (args["rsh"] != null)
            {
                //Чтение истории СМО
                ReadSMOHistory(args);
                return;
            }
            PrintUsage();
        }

        /// <summary>
        /// Данный метод выполняет чтение истории СМО карты полиса ОМС.
        /// Для успешного выполнения данного метода требуется модуль безопасности 
        /// с соответствующими ключами шифрования разрешающими выполнение данной операции
        /// </summary>
        /// <param name="args">Параметры запуска приложения</param>
        static void ReadSMOHistory(ArgumentCollection args)
        {
            printf();
            printf("Чтение истории СМО");
            if (string.IsNullOrEmpty(args["rsh"]))
            {
                printf("Не задано имя устройства чтения смарт-карт для работы с полисом ОМС");
                return;
            }
            if (args.ContainsKey("token") == false || string.IsNullOrEmpty(args["token"]))
            {
                printf("Не задано имя устройства чтения смарт-карт для работы с модулем безопасности");
                return;
            }
            //Чтение информации о владельце карты полиса ОМС
            PCSCReadersManager manager = new PCSCReadersManager();
            try
            {
                //Установление соединения с смарт-карт API
                manager.EstablishContext(SmartCard.PCSC.READERSCONTEXTSCOPE.SCOPE_USER);
                //Вывод списка ридеров в консоль
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["rsh"]) == false)
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                    return;
                }
                if (manager.OfType<ISCard>().Select(s => s.ReaderName).ToList().Contains(args["token"]) == false)
                {
                    printf("Устройство чтения смарт-карт с именем [{0}] не найдено в системе.");
                    manager.ReleaseContext();
                    return;
                }
                //Получение объекта ридера
                ISCard policy_card = manager[args["rsh"]];
                ISCard token_card = manager[args["token"]];
                //Создание объекта для работы с картой полиса ОМС
                PolicySmartcard policy = new PolicySmartcard(policy_card, token_card);
                //Подключение к карте полиса ОМС
                policy.Connect();
                //Аутентификация с картой модуля безопасности
                policy.Authentificate_External("11111111", true);
                //Чтение истории СМО
                List<SMOInformation> history_list = policy.GetSMOHistory();
                printf("Количество элементов истории: {0}", history_list.Count);
                history_list.ForEach(smo =>
                {
                    printf("Информация о предыдущем СМО:");
                    if (smo != null)
                    {
                        printf("Данные страхового приложения:");
                        printf("{0} = {1}", "ОГРН".PadRight(35, ' '), smo.OGRN);
                        printf("{0} = {1}", "ОКАТО".PadRight(35, ' '), smo.OKATO);
                        printf("{0} = {1}", "Дата начала страхования".PadRight(35, ' '), FormatPolicyDate(smo.InsuranceStartDate, "Отсутствует"));
                        printf("{0} = {1}", "Окончание срока действия".PadRight(35, ' '), FormatPolicyDate(smo.InsuranceExpireDate, "Не ограничено"));
                        if (smo.EDS != null)
                        {
                            printf("Данные ЭЦП");
                            printf("{0} = {1}", "Сертификат".PadRight(35, ' '), new X509Certificate2(smo.EDS.Certificate).Subject);
                            printf("{0} = {1}", "ЭЦП".PadRight(35, ' '), smo.EDS.Signature.ToHexString());
                        }
                        else
                        {
                            printf("{0} = {1}", "Данные ЭЦП".PadRight(35, ' '), "Отсутствуют");
                        }
                    }
                });
                //Отключение от карты полиса ОМС
                policy.Disconnect();
                //Отключение от смарт-карт API
                manager.ReleaseContext();
                return;
            }
            catch (Exception exception)
            {
                printf("Ошибка чтения информации с карты:\r\n", exception.Message);
            }
        }

        /// <summary>
        /// Вывод информации о параметрах запуска приложения
        /// </summary>
        static void PrintUsage()
        {
            Console.Clear();
            printf("ЗАО Атлас-Карт © 2011. Пример работы с картой полиса ОМС.");
            printf("-rdl | Получение списка ридеров в системе");
            printf("-rsh=\"...\" -token=\"...\" | Получение списка истории СМО. Пример: -rsh=\"ACS ACR38U 0\" -token=\"\" ");
        }

        /// <summary>
        /// Вывод информационной строки в консоль
        /// </summary>
        /// <param name="format">Форматирующая строка</param>
        /// <param name="parameters">Параметры</param>
        static void printf(string format = "", params object[] parameters)
        {
            Console.WriteLine(format, parameters);
        }

        /// <summary>
        /// Форматирование строки для вывода в консоль
        /// </summary>
        /// <param name="value">Значение строки</param>
        /// <param name="null_value">Значение по умолчанию ( если значение выводимой строки равно 'null')</param>
        /// <returns>Строку для вывода в консоль</returns>
        static string FormatPolicyText(string value, string null_value)
        {
            if (string.IsNullOrEmpty(value))
                return null_value;
            return value;
        }

        /// <summary>
        /// Форматирование даты для вывода в консоль
        /// </summary>
        /// <param name="date">Значение даты</param>
        /// <param name="null_value">Значение по умолчанию ( если значение выводимой даты равно 'null')</param>
        /// <returns>Строку для вывода в консоль</returns>
        static string FormatPolicyDate(DateTime? date, string null_value)
        {
            if (date.HasValue == false)
                return null_value;
            if (date.HasValue)
            {
                string sdate = date.Value.ToString("ddMMyyyy");
                if (sdate == "01011900")
                    return null_value;
                else
                    return date.Value.ToString("dd.MM.yyyy");
            }
            return null_value;
        }
    }
}